<!DOCTYPE html>
<html lang="{{ str_replace('_', '-', app()->getLocale()) }}">
    <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <title>Shop - Beauty Bawl</title>
        <meta name="description" content="Shop premium beauty products for personal and family care. Browse skincare, makeup, haircare, and beauty tools with secure checkout." />
        
        <link rel="preconnect" href="https://fonts.bunny.net">
        <link href="https://fonts.bunny.net/css?family=instrument-sans:400,500,600" rel="stylesheet" />
        
        @if (file_exists(public_path('build/manifest.json')) || file_exists(public_path('hot')))
            @vite(['resources/css/app.css', 'resources/js/app.js'])
        @else
            <script src="https://cdn.tailwindcss.com"></script>
            <script>
                tailwind.config = {
                    darkMode: 'class',
                    theme: {
                        extend: {
                            colors: {
                                primary: {
                                    50: '#fdf2f8',
                                    100: '#fce7f3',
                                    500: '#ec4899',
                                    600: '#db2777',
                                    700: '#be185d',
                                }
                            }
                        }
                    }
                }
            </script>
        @endif
    </head>
    <body class="bg-gradient-to-br from-pink-50 via-white to-purple-50 dark:bg-gradient-to-br dark:from-gray-900 dark:via-gray-800 dark:to-gray-900 text-gray-900 dark:text-gray-100 antialiased">
        
        @include('partials.header')

        <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            <!-- Breadcrumb -->
            <nav class="flex mb-8 text-sm" aria-label="Breadcrumb">
                <ol class="inline-flex items-center space-x-2">
                    <li><a href="/" class="text-gray-500 hover:text-pink-600">Home</a></li>
                    <li><span class="text-gray-400">/</span></li>
                    <li class="text-gray-900 dark:text-gray-100 font-medium">Shop</li>
                </ol>
            </nav>

            <!-- Page Header -->
            <div class="mb-8">
                @php
                    $pageTitle = 'Shop All Products';
                    $pageDescription = 'Discover our curated collection of premium beauty products';
                    
                    if (isset($category)) {
                        $pageTitle = ucfirst($category) . ' Products';
                        $pageDescription = 'Browse our selection of ' . strtolower($category) . ' products';
                    } elseif (isset($brand)) {
                        $pageTitle = ucfirst($brand) . ' Products';
                        $pageDescription = 'Shop ' . ucfirst($brand) . ' beauty products';
                    } elseif (isset($filter)) {
                        if ($filter === 'featured') {
                            $pageTitle = 'Featured Products';
                            $pageDescription = 'Our hand-picked collection of the best beauty products';
                        } elseif ($filter === 'deals') {
                            $pageTitle = '🔥 Hot Deals';
                            $pageDescription = 'Amazing discounts on premium beauty products';
                        }
                    } elseif (isset($search)) {
                        $pageTitle = 'Search Results: ' . $search;
                        $pageDescription = 'Products matching "' . $search . '"';
                    }
                @endphp
                
                <h1 class="text-4xl font-bold mb-3">{{ $pageTitle }}</h1>
                <p class="text-gray-600 dark:text-gray-400">{{ $pageDescription }}</p>
            </div>

            <div class="grid lg:grid-cols-4 gap-8">
                <!-- Filters Sidebar -->
                <aside class="lg:col-span-1">
                    <div class="bg-white dark:bg-gray-800 rounded-2xl shadow-lg p-6 sticky top-24">
                        <h2 class="text-xl font-bold mb-6 flex items-center gap-2">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2.586a1 1 0 01-.293.707l-6.414 6.414a1 1 0 00-.293.707V17l-4 4v-6.586a1 1 0 00-.293-.707L3.293 7.293A1 1 0 013 6.586V4z" />
                            </svg>
                            Filters
                        </h2>

                        <!-- Categories -->
                        <div class="mb-6">
                            <h3 class="font-semibold mb-3">Categories</h3>
                            <div class="space-y-2">
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="category" value="all" class="w-4 h-4 text-pink-600 rounded" checked>
                                    <span class="text-sm">All Products</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="category" value="skincare" class="w-4 h-4 text-pink-600 rounded">
                                    <span class="text-sm">Skincare</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="category" value="haircare" class="w-4 h-4 text-pink-600 rounded">
                                    <span class="text-sm">Haircare</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="category" value="makeup" class="w-4 h-4 text-pink-600 rounded">
                                    <span class="text-sm">Makeup</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="category" value="tools" class="w-4 h-4 text-pink-600 rounded">
                                    <span class="text-sm">Tools & Devices</span>
                                </label>
                            </div>
                        </div>

                        <!-- Price Range -->
                        <div class="mb-6 pb-6 border-b border-gray-200 dark:border-gray-700">
                            <h3 class="font-semibold mb-3">Price Range</h3>
                            <div class="space-y-2">
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="radio" name="price" value="under-20" class="w-4 h-4 text-pink-600">
                                    <span class="text-sm">Under $20</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="radio" name="price" value="20-50" class="w-4 h-4 text-pink-600">
                                    <span class="text-sm">$20 - $50</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="radio" name="price" value="50-100" class="w-4 h-4 text-pink-600">
                                    <span class="text-sm">$50 - $100</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="radio" name="price" value="over-100" class="w-4 h-4 text-pink-600">
                                    <span class="text-sm">Over $100</span>
                                </label>
                            </div>
                        </div>

                        <!-- Special Features -->
                        <div class="mb-6">
                            <h3 class="font-semibold mb-3">Special Features</h3>
                            <div class="space-y-2">
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="feature" value="new" class="w-4 h-4 text-pink-600 rounded">
                                    <span class="text-sm">✨ New Arrivals</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="feature" value="sale" class="w-4 h-4 text-pink-600 rounded">
                                    <span class="text-sm">🔥 On Sale</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="feature" value="eco" class="w-4 h-4 text-pink-600 rounded">
                                    <span class="text-sm">🌿 Eco-Friendly</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer hover:text-pink-600">
                                    <input type="checkbox" name="feature" value="family" class="w-4 h-4 text-pink-600 rounded">
                                    <span class="text-sm">👶 Family Safe</span>
                                </label>
                            </div>
                        </div>

                        <button class="apply-filters-btn w-full px-4 py-3 bg-gradient-to-r from-pink-600 to-purple-600 text-white rounded-full font-medium hover:shadow-lg transition-all">
                            Apply Filters
                        </button>
                    </div>
                </aside>

                <!-- Products Grid -->
                <div class="lg:col-span-3">
                    <!-- Toolbar -->
                    <div class="flex flex-wrap items-center justify-between mb-6 gap-4">
                        <p class="product-count text-gray-600 dark:text-gray-400">Showing <strong>12</strong> products</p>
                        
                        <div class="flex items-center gap-4">
                            <label class="flex items-center gap-2 text-sm">
                                <span>Sort by:</span>
                                <select class="px-4 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg">
                                    <option value="featured">Featured</option>
                                    <option value="price-low">Price: Low to High</option>
                                    <option value="price-high">Price: High to Low</option>
                                    <option value="newest">Newest</option>
                                    <option value="best-selling">Best Selling</option>
                                </select>
                            </label>

                            <div class="flex gap-2">
                                <button class="grid-view-btn p-2 bg-pink-600 text-white rounded-lg" title="Grid view">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2V6zM14 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V6zM4 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2v-2zM14 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2v-2z" />
                                    </svg>
                                </button>
                                <button class="list-view-btn p-2 bg-gray-200 dark:bg-gray-700 rounded-lg" title="List view">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16" />
                                    </svg>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Products Grid -->
                    <div class="products-grid grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
                        @foreach (range(1, 12) as $i)
                            @php
                                $categories = ['skincare', 'haircare', 'makeup', 'tools'];
                                $category = $categories[($i - 1) % 4];
                                $price = 15 + $i * 5;
                                $features = [];
                                if ($i % 3 == 0) $features[] = 'new';
                                if ($i % 2 == 0) $features[] = 'sale';
                                if ($i % 4 == 0) $features[] = 'eco';
                                $featuresStr = implode(',', $features);
                            @endphp
                            <article class="product-card group bg-white dark:bg-gray-800 rounded-2xl shadow-lg hover:shadow-2xl transform hover:-translate-y-2 transition-all duration-300 overflow-hidden" 
                                data-id="{{ $i }}" 
                                data-category="{{ $category }}" 
                                data-price="{{ $price }}"
                                data-features="{{ $featuresStr }}"
                                data-sales="{{ rand(50, 500) }}">
                                <a href="/product/{{ $i }}" class="block">
                                    <div class="relative aspect-square bg-gradient-to-br from-gray-100 to-gray-200 dark:from-gray-700 dark:to-gray-600">
                                        <div class="absolute inset-0 flex items-center justify-center">
                                            <svg class="w-24 h-24 text-gray-300 dark:text-gray-600" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1" d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z" />
                                            </svg>
                                        </div>
                                        
                                        @if($i % 3 == 0)
                                            <span class="absolute top-3 left-3 px-3 py-1 bg-pink-600 text-white text-xs font-bold rounded-full">NEW</span>
                                        @endif
                                        @if($i % 2 == 0)
                                            <span class="absolute top-3 right-3 px-3 py-1 bg-red-600 text-white text-xs font-bold rounded-full">-20%</span>
                                        @endif
                                        
                                        <button class="absolute bottom-3 right-3 w-10 h-10 bg-white/90 dark:bg-gray-800/90 backdrop-blur-sm rounded-full flex items-center justify-center hover:scale-110 transition-transform">
                                            <svg class="w-5 h-5 text-gray-600 dark:text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                                            </svg>
                                        </button>
                                    </div>

                                    <div class="p-5">
                                        <div class="flex items-center gap-1 mb-2">
                                            @for($star = 0; $star < 5; $star++)
                                                <svg class="w-4 h-4 {{ $star < 4 ? 'text-yellow-400' : 'text-gray-300' }}" fill="currentColor" viewBox="0 0 20 20">
                                                    <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                                </svg>
                                            @endfor
                                            <span class="text-xs text-gray-500 ml-1">({{ 15 + $i * 3 }})</span>
                                        </div>

                                        <h3 class="font-bold text-gray-900 dark:text-gray-100 mb-1 line-clamp-2 group-hover:text-pink-600 transition-colors">
                                            Premium Beauty Product {{ $i }}
                                        </h3>
                                        
                                        <p class="text-sm text-gray-500 dark:text-gray-400 mb-3">{{ ['Skincare', 'Haircare', 'Makeup', 'Tools'][($i - 1) % 4] }}</p>

                                        <div class="flex items-center justify-between">
                                            <div class="flex items-baseline gap-2">
                                                <span class="text-2xl font-bold text-pink-600">
                                                    ${{ sprintf('%.2f', 15 + $i * 5) }}
                                                </span>
                                                @if($i % 2 == 0)
                                                    <span class="text-sm text-gray-400 line-through">${{ sprintf('%.2f', 20 + $i * 5) }}</span>
                                                @endif
                                            </div>
                                            
                                            <button class="w-10 h-10 bg-gradient-to-r from-pink-600 to-purple-600 text-white rounded-full flex items-center justify-center hover:scale-110 hover:shadow-lg transition-all">
                                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4" />
                                                </svg>
                                            </button>
                                        </div>
                                    </div>
                                </a>
                            </article>
                        @endforeach
                    </div>

                    <!-- Pagination -->
                    <div class="flex items-center justify-between">
                        <p class="text-sm text-gray-600 dark:text-gray-400">Showing 1-12 of 48 products</p>
                        
                        <nav class="flex items-center gap-2">
                            <button class="px-4 py-2 bg-gray-200 dark:bg-gray-700 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600" disabled>
                                Previous
                            </button>
                            <button class="px-4 py-2 bg-gradient-to-r from-pink-600 to-purple-600 text-white rounded-lg font-medium">1</button>
                            <button class="px-4 py-2 bg-gray-200 dark:bg-gray-700 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600">2</button>
                            <button class="px-4 py-2 bg-gray-200 dark:bg-gray-700 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600">3</button>
                            <button class="px-4 py-2 bg-gray-200 dark:bg-gray-700 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600">4</button>
                            <button class="px-4 py-2 bg-gray-200 dark:bg-gray-700 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600">
                                Next
                            </button>
                        </nav>
                    </div>
                </div>
            </div>
        </main>

        @include('partials.footer')

        <script>
            // Filter functionality
            document.addEventListener('DOMContentLoaded', function() {
                const filterButton = document.querySelector('.apply-filters-btn');
                const sortSelect = document.querySelector('select');
                const productGrid = document.querySelector('.products-grid');
                const products = Array.from(document.querySelectorAll('.product-card'));

                // Apply Filters
                if (filterButton) {
                    filterButton.addEventListener('click', function() {
                        applyFilters();
                    });
                }

                // Sort functionality
                if (sortSelect) {
                    sortSelect.addEventListener('change', function() {
                        sortProducts(this.value);
                    });
                }

                // Category filter checkboxes
                const categoryCheckboxes = document.querySelectorAll('input[name="category"]');
                categoryCheckboxes.forEach(checkbox => {
                    checkbox.addEventListener('change', applyFilters);
                });

                // Price filter radio buttons
                const priceRadios = document.querySelectorAll('input[name="price"]');
                priceRadios.forEach(radio => {
                    radio.addEventListener('change', applyFilters);
                });

                // Feature checkboxes
                const featureCheckboxes = document.querySelectorAll('input[name="feature"]');
                featureCheckboxes.forEach(checkbox => {
                    checkbox.addEventListener('change', applyFilters);
                });

                function applyFilters() {
                    const selectedCategories = Array.from(document.querySelectorAll('input[name="category"]:checked'))
                        .map(cb => cb.value);
                    const selectedPrice = document.querySelector('input[name="price"]:checked')?.value;
                    const selectedFeatures = Array.from(document.querySelectorAll('input[name="feature"]:checked'))
                        .map(cb => cb.value);

                    let visibleCount = 0;

                    products.forEach(product => {
                        let show = true;

                        // Category filter
                        if (selectedCategories.length > 0 && !selectedCategories.includes('all')) {
                            const productCategory = product.dataset.category;
                            if (!selectedCategories.includes(productCategory)) {
                                show = false;
                            }
                        }

                        // Price filter
                        if (selectedPrice && show) {
                            const productPrice = parseFloat(product.dataset.price);
                            switch(selectedPrice) {
                                case 'under-20':
                                    if (productPrice >= 20) show = false;
                                    break;
                                case '20-50':
                                    if (productPrice < 20 || productPrice > 50) show = false;
                                    break;
                                case '50-100':
                                    if (productPrice < 50 || productPrice > 100) show = false;
                                    break;
                                case 'over-100':
                                    if (productPrice < 100) show = false;
                                    break;
                            }
                        }

                        // Feature filter
                        if (selectedFeatures.length > 0 && show) {
                            const productFeatures = (product.dataset.features || '').split(',');
                            const hasAllFeatures = selectedFeatures.every(feature => 
                                productFeatures.includes(feature)
                            );
                            if (!hasAllFeatures) show = false;
                        }

                        // Show/hide product
                        if (show) {
                            product.style.display = '';
                            visibleCount++;
                        } else {
                            product.style.display = 'none';
                        }
                    });

                    // Update product count
                    updateProductCount(visibleCount);
                }

                function sortProducts(sortValue) {
                    const sortedProducts = [...products];

                    switch(sortValue) {
                        case 'price-low':
                            sortedProducts.sort((a, b) => 
                                parseFloat(a.dataset.price) - parseFloat(b.dataset.price)
                            );
                            break;
                        case 'price-high':
                            sortedProducts.sort((a, b) => 
                                parseFloat(b.dataset.price) - parseFloat(a.dataset.price)
                            );
                            break;
                        case 'newest':
                            sortedProducts.sort((a, b) => 
                                parseInt(b.dataset.id) - parseInt(a.dataset.id)
                            );
                            break;
                        case 'best-selling':
                            sortedProducts.sort((a, b) => 
                                parseInt(b.dataset.sales || 0) - parseInt(a.dataset.sales || 0)
                            );
                            break;
                        default: // featured
                            // Keep original order
                            break;
                    }

                    // Re-append products in sorted order
                    if (sortValue !== 'featured') {
                        sortedProducts.forEach(product => {
                            productGrid.appendChild(product);
                        });
                    }
                }

                function updateProductCount(count) {
                    const countElement = document.querySelector('.product-count');
                    if (countElement) {
                        countElement.innerHTML = `Showing <strong>${count}</strong> products`;
                    }
                }

                // View toggle
                const gridViewBtn = document.querySelector('.grid-view-btn');
                const listViewBtn = document.querySelector('.list-view-btn');

                if (gridViewBtn && listViewBtn) {
                    gridViewBtn.addEventListener('click', function() {
                        productGrid.classList.remove('list-view');
                        gridViewBtn.classList.add('bg-pink-600', 'text-white');
                        gridViewBtn.classList.remove('bg-gray-200', 'dark:bg-gray-700');
                        listViewBtn.classList.remove('bg-pink-600', 'text-white');
                        listViewBtn.classList.add('bg-gray-200', 'dark:bg-gray-700');
                    });

                    listViewBtn.addEventListener('click', function() {
                        productGrid.classList.add('list-view');
                        listViewBtn.classList.add('bg-pink-600', 'text-white');
                        listViewBtn.classList.remove('bg-gray-200', 'dark:bg-gray-700');
                        gridViewBtn.classList.remove('bg-pink-600', 'text-white');
                        gridViewBtn.classList.add('bg-gray-200', 'dark:bg-gray-700');
                    });
                }
            });
        </script>

        <style>
            .list-view {
                display: flex !important;
                flex-direction: column !important;
            }
            .list-view .product-card {
                display: flex !important;
                flex-direction: row !important;
                max-width: 100% !important;
            }
            .list-view .product-card > a {
                display: flex !important;
                flex-direction: row !important;
                width: 100% !important;
            }
            .list-view .product-card .aspect-square {
                width: 200px !important;
                height: 200px !important;
                flex-shrink: 0 !important;
            }
        </style>
    </body>
</html>
