<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     * Creates PostgreSQL triggers to automatically manage commissions
     * when completions are updated/deleted directly in the database.
     */
    public function up(): void
    {
        // Function: Delete commissions when completion approved changes to false
        DB::unprepared("
            CREATE OR REPLACE FUNCTION delete_commissions_on_unapproval()
            RETURNS TRIGGER AS $$
            BEGIN
                -- If approved changed from true to false, delete commissions
                IF OLD.approved = true AND NEW.approved = false THEN
                    DELETE FROM commissions WHERE completions_id = NEW.completions_id;
                END IF;
                RETURN NEW;
            END;
            $$ LANGUAGE plpgsql;
        ");

        // Function: Create/Update commissions when completion is updated
        DB::unprepared("
            CREATE OR REPLACE FUNCTION manage_commissions_on_completion_update()
            RETURNS TRIGGER AS $$
            DECLARE
                service_commission_rate DECIMAL(5,2);
                new_commission_amount DECIMAL(10,2);
            BEGIN
                -- Get service commission rate
                SELECT commission INTO service_commission_rate
                FROM services
                WHERE service_id = NEW.service_id;

                -- If approved changed from false to true, create commission
                IF OLD.approved = false AND NEW.approved = true AND NEW.charged > 0 THEN
                    new_commission_amount := (NEW.charged * service_commission_rate) / 100;
                    
                    -- Only insert if commission doesn't exist
                    INSERT INTO commissions (completions_id, user_id, commission_amount, created_at)
                    SELECT NEW.completions_id, NEW.user_id, new_commission_amount, NOW()
                    WHERE NOT EXISTS (
                        SELECT 1 FROM commissions WHERE completions_id = NEW.completions_id
                    );
                    
                -- If approved=true and charged/service changed, update commission
                ELSIF NEW.approved = true AND (OLD.charged != NEW.charged OR OLD.service_id != NEW.service_id) THEN
                    new_commission_amount := (NEW.charged * service_commission_rate) / 100;
                    
                    UPDATE commissions
                    SET commission_amount = new_commission_amount,
                        user_id = NEW.user_id
                    WHERE completions_id = NEW.completions_id;
                    
                    -- Create if doesn't exist
                    IF NOT FOUND AND NEW.charged > 0 THEN
                        INSERT INTO commissions (completions_id, user_id, commission_amount, created_at)
                        VALUES (NEW.completions_id, NEW.user_id, new_commission_amount, NOW());
                    END IF;
                END IF;
                
                RETURN NEW;
            END;
            $$ LANGUAGE plpgsql;
        ");

        // Function: Create commission when completion is inserted with approved=true
        DB::unprepared("
            CREATE OR REPLACE FUNCTION create_commission_on_completion_insert()
            RETURNS TRIGGER AS $$
            DECLARE
                service_commission_rate DECIMAL(5,2);
                commission_amount DECIMAL(10,2);
            BEGIN
                -- If completion is approved, create commission
                IF NEW.approved = true AND NEW.charged > 0 THEN
                    SELECT commission INTO service_commission_rate
                    FROM services
                    WHERE service_id = NEW.service_id;
                    
                    commission_amount := (NEW.charged * service_commission_rate) / 100;
                    
                    INSERT INTO commissions (completions_id, user_id, commission_amount, created_at)
                    VALUES (NEW.completions_id, NEW.user_id, commission_amount, NOW());
                END IF;
                
                RETURN NEW;
            END;
            $$ LANGUAGE plpgsql;
        ");

        // Function: Delete commissions when completion is deleted
        DB::unprepared("
            CREATE OR REPLACE FUNCTION delete_commissions_on_completion_delete()
            RETURNS TRIGGER AS $$
            BEGIN
                DELETE FROM commissions WHERE completions_id = OLD.completions_id;
                RETURN OLD;
            END;
            $$ LANGUAGE plpgsql;
        ");

        // Create triggers
        DB::unprepared("
            CREATE TRIGGER trigger_delete_commissions_on_unapproval
            AFTER UPDATE ON completions
            FOR EACH ROW
            WHEN (OLD.approved IS DISTINCT FROM NEW.approved)
            EXECUTE FUNCTION delete_commissions_on_unapproval();
        ");

        DB::unprepared("
            CREATE TRIGGER trigger_manage_commissions_on_update
            AFTER UPDATE ON completions
            FOR EACH ROW
            EXECUTE FUNCTION manage_commissions_on_completion_update();
        ");

        DB::unprepared("
            CREATE TRIGGER trigger_create_commission_on_insert
            AFTER INSERT ON completions
            FOR EACH ROW
            EXECUTE FUNCTION create_commission_on_completion_insert();
        ");

        DB::unprepared("
            CREATE TRIGGER trigger_delete_commissions_on_delete
            BEFORE DELETE ON completions
            FOR EACH ROW
            EXECUTE FUNCTION delete_commissions_on_completion_delete();
        ");
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Drop triggers
        DB::unprepared('DROP TRIGGER IF EXISTS trigger_delete_commissions_on_unapproval ON completions');
        DB::unprepared('DROP TRIGGER IF EXISTS trigger_manage_commissions_on_update ON completions');
        DB::unprepared('DROP TRIGGER IF EXISTS trigger_create_commission_on_insert ON completions');
        DB::unprepared('DROP TRIGGER IF EXISTS trigger_delete_commissions_on_delete ON completions');

        // Drop functions
        DB::unprepared('DROP FUNCTION IF EXISTS delete_commissions_on_unapproval()');
        DB::unprepared('DROP FUNCTION IF EXISTS manage_commissions_on_completion_update()');
        DB::unprepared('DROP FUNCTION IF EXISTS create_commission_on_completion_insert()');
        DB::unprepared('DROP FUNCTION IF EXISTS delete_commissions_on_completion_delete()');
    }
};
