<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = User::query();

        // Filter by role
        if ($request->has('role')) {
            $query->where('role', $request->role);
        }

        // Filter by designation
        if ($request->has('designation')) {
            $query->where('designation', $request->designation);
        }

        // Search by name
        if ($request->has('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        $users = $query->orderBy('name')->get();

        return response()->json([
            'success' => true,
            'data' => $users->map(fn($user) => [
                'user_id' => $user->user_id,
                'name' => $user->name,
                'role' => $user->role,
                'designation' => $user->designation,
                'code' => $user->code,
                'basic_pay' => $user->basic_pay,
                'created_at' => $user->created_at,
            ]),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'role' => 'required|in:admin,employee',
            'designation' => 'nullable|string|max:255',
            'code' => 'required|string|unique:users,code',
            'basic_pay' => 'nullable|numeric|min:0',
            'password' => 'required|string|min:4',
        ]);

        $user = User::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'User created successfully',
            'data' => [
                'user_id' => $user->user_id,
                'name' => $user->name,
                'role' => $user->role,
                'designation' => $user->designation,
                'code' => $user->code,
                'basic_pay' => $user->basic_pay,
            ],
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $user = User::findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => [
                'user_id' => $user->user_id,
                'name' => $user->name,
                'role' => $user->role,
                'designation' => $user->designation,
                'code' => $user->code,
                'basic_pay' => $user->basic_pay,
                'created_at' => $user->created_at,
            ],
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'role' => 'sometimes|in:admin,employee',
            'designation' => 'nullable|string|max:255',
            'code' => 'sometimes|string|unique:users,code,' . $id . ',user_id',
            'basic_pay' => 'nullable|numeric|min:0',
            'password' => 'sometimes|string|min:4',
        ]);

        $user->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'User updated successfully',
            'data' => [
                'user_id' => $user->user_id,
                'name' => $user->name,
                'role' => $user->role,
                'designation' => $user->designation,
                'code' => $user->code,
                'basic_pay' => $user->basic_pay,
            ],
        ]);
    }

    /**
     * Remove the specified resource from storage (soft delete).
     * The user's code will be released for reuse.
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $userName = $user->name;
        $userCode = $user->code;

        // Soft delete the user (code is automatically released)
        $user->delete();

        return response()->json([
            'success' => true,
            'message' => "User '{$userName}' deleted successfully. Code '{$userCode}' is now available for reuse.",
            'released_code' => $userCode,
        ]);
    }

    /**
     * Get all soft deleted users.
     */
    public function trashed()
    {
        $trashedUsers = User::onlyTrashed()->orderBy('deleted_at', 'desc')->get();

        return response()->json([
            'success' => true,
            'data' => $trashedUsers->map(fn($user) => [
                'user_id' => $user->user_id,
                'name' => $user->name,
                'role' => $user->role,
                'designation' => $user->designation,
                'code' => $user->code,
                'basic_pay' => $user->basic_pay,
                'deleted_at' => $user->deleted_at,
            ]),
        ]);
    }

    /**
     * Restore a soft deleted user with a new code.
     */
    public function restore(Request $request, $id)
    {
        $user = User::onlyTrashed()->findOrFail($id);

        // Validate the new code
        $validated = $request->validate([
            'code' => 'required|string|max:20',
        ]);

        // Check if the new code is already taken by an active user
        $codeExists = User::where('code', $validated['code'])
            ->whereNull('deleted_at')
            ->exists();

        if ($codeExists) {
            return response()->json([
                'success' => false,
                'message' => "Code '{$validated['code']}' is already in use. Please choose a different code.",
            ], 422);
        }

        // Update the code and restore the user
        $user->code = $validated['code'];
        $user->restore();

        return response()->json([
            'success' => true,
            'message' => "User '{$user->name}' restored successfully with code '{$user->code}'",
            'data' => [
                'user_id' => $user->user_id,
                'name' => $user->name,
                'role' => $user->role,
                'designation' => $user->designation,
                'code' => $user->code,
                'basic_pay' => $user->basic_pay,
            ],
        ]);
    }

    /**
     * Permanently delete a soft deleted user.
     */
    public function forceDelete($id)
    {
        $user = User::onlyTrashed()->findOrFail($id);
        $userName = $user->name;

        $user->forceDelete();

        return response()->json([
            'success' => true,
            'message' => "User '{$userName}' permanently deleted from database.",
        ]);
    }
}
