<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Service;
use App\Models\Completion;
use App\Models\Commission;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class TestCommissionDeletion extends Command
{
    protected $signature = 'test:commission-deletion';
    protected $description = 'Test automatic commission deletion when completion is deleted or unapproved';

    public function handle(): int
    {
        $this->info('🧪 Testing Automatic Commission Deletion...');
        $this->newLine();

        DB::beginTransaction();

        try {
            // Get test data
            $service = Service::first();
            $user = User::first();

            if (!$service || !$user) {
                $this->error('❌ No service or user found. Please seed database first.');
                DB::rollBack();
                return self::FAILURE;
            }

            // Test 1: Deletion of completion deletes commission
            $this->info('Test 1: Delete completion → Commission should be deleted');
            $this->testCompletionDeletion($service, $user);
            $this->newLine();

            // Test 2: Changing approved from true to false deletes commission
            $this->info('Test 2: Change approved to false → Commission should be deleted');
            $this->testApprovalChange($service, $user);
            $this->newLine();

            // Test 3: Creating approved completion creates commission
            $this->info('Test 3: Create approved completion → Commission created');
            $this->testApprovedCompletion($service, $user);
            $this->newLine();

            $this->info('✅ All tests completed successfully!');

            DB::rollBack(); // Rollback all test data
            return self::SUCCESS;
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error('❌ Test failed: ' . $e->getMessage());
            return self::FAILURE;
        }
    }

    private function testCompletionDeletion($service, $user): void
    {
        // Create completion and commission
        $completion = Completion::create([
            'service_id' => $service->service_id,
            'user_id' => $user->user_id,
            'charged' => 100.00,
            'date' => now(),
            'approved' => true,
        ]);

        $commission = Commission::create([
            'completions_id' => $completion->completions_id,
        ]);

        $commissionId = $commission->commission_id;
        $this->line("   Created Commission ID: {$commissionId}");

        // Delete completion
        $completion->delete();

        // Check if commission was deleted
        $stillExists = Commission::find($commissionId);

        if (!$stillExists) {
            $this->info('   ✓ Commission automatically deleted when completion was deleted');
        } else {
            $this->error('   ✗ Commission still exists after completion deletion');
        }
    }

    private function testApprovalChange($service, $user): void
    {
        // Create approved completion with commission
        $completion = Completion::create([
            'service_id' => $service->service_id,
            'user_id' => $user->user_id,
            'charged' => 200.00,
            'date' => now(),
            'approved' => true,
        ]);

        $commission = Commission::create([
            'completions_id' => $completion->completions_id,
        ]);

        $commissionId = $commission->commission_id;
        $this->line("   Created Commission ID: {$commissionId}");
        $this->line("   Completion approved: true");

        // Change approved to false
        $completion->approved = false;
        $completion->save();

        $this->line("   Changed approval to: false");

        // Check if commission was deleted
        $stillExists = Commission::find($commissionId);

        if (!$stillExists) {
            $this->info('   ✓ Commission automatically deleted when approval changed to false');
        } else {
            $this->error('   ✗ Commission still exists after approval change');
        }

        // Cleanup
        $completion->delete();
    }

    private function testApprovedCompletion($service, $user): void
    {
        // Create approved completion
        $completion = Completion::create([
            'service_id' => $service->service_id,
            'user_id' => $user->user_id,
            'charged' => 150.00,
            'date' => now(),
            'approved' => true,
        ]);

        $this->line("   Created Completion ID: {$completion->completions_id}");

        // Create commission
        $commission = Commission::create([
            'completions_id' => $completion->completions_id,
        ]);

        $this->line("   Created Commission ID: {$commission->commission_id}");
        $this->line("   Commission Amount: \${$commission->commission_amount}");

        if ($commission->commission_amount > 0) {
            $this->info('   ✓ Commission created and calculated correctly');
        } else {
            $this->warn('   ⚠️  Commission amount is 0 - check observer logic');
        }

        // Cleanup
        $commission->delete();
        $completion->delete();
    }
}
